package com.zksy.szpt.filter;


import cn.hutool.crypto.digest.DigestUtil;
import cn.hutool.json.JSONUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.zksy.szpt.domain.HttpResult;
import com.zksy.szpt.domain.HttpResultState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;

import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.annotation.Resource;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Map;


@Order(1)
@WebFilter(urlPatterns = "/**", filterName = "SignatureVerificationFilter")
@Component
public class SignatureVerificationFilter extends OncePerRequestFilter {

    public static Logger logger = LoggerFactory.getLogger(SignatureVerificationFilter.class);

    private final ObjectMapper objectMapper;

    public SignatureVerificationFilter(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        // 对request进行包装，支持重复读取body
        ReusableBodyRequestWrapper requestWrapper = new ReusableBodyRequestWrapper(request);
        // 校验签名
        if (this.verifySignature(requestWrapper, response)) {
            filterChain.doFilter(requestWrapper, response);
        }
    }


    // 签名秘钥
    private String secretKey = "2";

    /**
     * 校验签名
     *
     * @param request  HTTP请求
     * @param response HTTP响应
     * @return 签名验证结果
     * @throws IOException 如果读取请求体失败
     */
    public boolean verifySignature(HttpServletRequest request, HttpServletResponse response) throws IOException {
        // 签名
        String appId = request.getHeader("appId");
        // 签名
        String sign = request.getHeader("signature");
        // 随机数
        String nonce = request.getHeader("nonce");
        // 时间戳
        String timestampStr = request.getHeader("timestamp");

        if (!StringUtils.hasText(sign) || !StringUtils.hasText(nonce) || !StringUtils.hasText(timestampStr)) {
            this.write(response, "参数错误");
            return false;
        }

        // timestamp 10分钟内有效
//        long timestamp = Long.parseLong(timestampStr);
//        long currentTimestamp = System.currentTimeMillis() / 1000;
//        if (Math.abs(currentTimestamp - timestamp) > 600) {
//            this.write(response, "请求已过期");
//            return false;
//        }

        // 防止请求重放，nonce只能用一次，放在redis中，有效期 20分钟
//        String nonceKey = "SignatureVerificationFilter:nonce:" + nonce;
//        if (!this.redisTemplate.opsForValue().setIfAbsent(nonceKey, "1", 20, TimeUnit.MINUTES)) {
//            this.write(response, "nonce无效");
//            return false;
//        }

        // 请求体
        String body = objectMapper.writeValueAsString(objectMapper.readValue(StreamUtils.copyToString(request.getInputStream(), StandardCharsets.UTF_8), Map.class));
        // 需要签名的数据：secretKey+noce+timestampStr+body
        // 校验签名
        String data = String.format("%s%s%s%s%s", appId, nonce, timestampStr, body, this.secretKey);
        if (!DigestUtil.md5Hex(data).equals(sign)) {
            write(response, "签名有误");
            return false;
        }
        return true;
    }

    /**
     * 向客户端写入响应信息
     *
     * @param response HTTP响应
     * @param msg      响应信息
     * @throws IOException 如果写入失败
     */
    private void write(HttpServletResponse response, String msg) throws IOException {
        HttpResultState httpResultState = HttpResultState.INVALID_TOKEN;
        httpResultState.setMessage(msg);
        HttpResult<String> httpResult = new HttpResult<>(httpResultState);
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.getWriter().write(JSONUtil.toJsonStr(httpResult));
    }
}

