package com.zksy.szpt.filter;

import javax.servlet.ReadListener;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import java.io.*;
import java.nio.charset.StandardCharsets;

/**
 * 该类用于包装HttpServletRequest，以便在读取请求体后仍可重复读取
 */
public class ReusableBodyRequestWrapper1 extends HttpServletRequestWrapper {

    //参数字节数组，用于存储请求体的字节数据
    private byte[] requestBody;

    /**
     * 构造函数，初始化包装类
     * @param request 原始HttpServletRequest对象
     * @throws IOException 如果读取请求体时发生IO错误
     */
    public ReusableBodyRequestWrapper1(HttpServletRequest request, String body) throws IOException {
        super(request);
        this.requestBody = body.getBytes(StandardCharsets.UTF_8);
    }

    // 构造方法：从请求中读取 body 并缓存
    public ReusableBodyRequestWrapper1(HttpServletRequest request) throws IOException {
        super(request);
        this.requestBody = request.getReader().lines()
                .reduce("", (accumulator, actual) -> accumulator + actual)
                .getBytes(StandardCharsets.UTF_8);
    }

    /**
     * 重写getInputStream方法，实现请求体的重复读取
     * @return 包含请求体数据的ServletInputStream对象
     * @throws IOException 如果读取请求体时发生IO错误
     */
    @Override
    public ServletInputStream getInputStream() throws IOException {
        //创建一个 ByteArrayInputStream 对象，用于重复读取requestBody
        final ByteArrayInputStream bais = new ByteArrayInputStream(requestBody);
        return new ServletInputStream() {

            @Override
            public boolean isFinished() {
                //始终返回false，表示数据流未完成
                return false;
            }

            @Override
            public boolean isReady() {
                //始终返回false，表示数据流未准备好
                return false;
            }

            @Override
            public void setReadListener(ReadListener listener) {
                //不执行任何操作，因为该数据流不支持异步操作
            }

            @Override
            public int read() {
                //从ByteArrayInputStream中读取数据
                return bais.read();
            }
        };
    }

    /**
     * 重写getReader方法，返回一个基于getInputStream的BufferedReader
     * @return 包含请求体数据的BufferedReader对象
     * @throws IOException 如果读取请求体时发生IO错误
     */
    @Override
    public BufferedReader getReader() throws IOException {
        //基于getInputStream创建BufferedReader
        return new BufferedReader(new InputStreamReader(this.getInputStream()));
    }

    // 获取请求体的字符串
    public String getBody() {
        return new String(requestBody, StandardCharsets.UTF_8);
    }
}