package com.zksy.szpt.filter;


import cn.hutool.crypto.digest.DigestUtil;
import cn.hutool.json.JSONUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.zksy.szpt.domain.CommonRequestDTO;
import com.zksy.szpt.domain.CurrentUserInfo;
import com.zksy.szpt.domain.HttpResult;
import com.zksy.szpt.domain.HttpResultState;
import com.zksy.szpt.domain.po.AppStore;
import com.zksy.szpt.service.AppStoreService;
import com.zksy.szpt.util.EncryptUtil;
import com.zksy.szpt.util.RedisKeyValidator;
import com.zksy.szpt.util.SignatureUtil;
import com.zksy.szpt.util.UserContextHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.annotation.Resource;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 请求body { appkey , method , hotelCode , content（根据密钥[md5]对dto的json加密aes）, sign(前面的参数一起哈希运算)}  。 无其他参数
 * 数据获取到json解密后转发到对应的controller
 */
//@Component(value = "signatureVerificationFilter1")
public class SignatureVerificationFilter1 extends OncePerRequestFilter {

    public Logger logger = LoggerFactory.getLogger(SignatureVerificationFilter1.class);

    @Resource
    ObjectMapper objectMapper;
    AppStore appStore;

    @Resource
    private RedisTemplate<String, Object> redisTemplate;

    private final AppStoreService appStoreService;

    public SignatureVerificationFilter1(AppStoreService appStoreService) {
        this.appStoreService = appStoreService;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        // 对request进行包装，支持重复读取body
        ReusableBodyRequestWrapper1 requestWrapper = new ReusableBodyRequestWrapper1(request);
        String body = requestWrapper.getBody();
        String result = this.verifySignature(body);
        // 校验签名
        if (result != null) {
            this.write(response, result);
        } else {
            CommonRequestDTO commonRequestDTO = objectMapper.readValue(body, CommonRequestDTO.class);
            String content = EncryptUtil.getInstance().AESDecode(commonRequestDTO.getContent(), appStore.getAppSecret());
            ReusableBodyRequestWrapper1 requestWrapper1 = new ReusableBodyRequestWrapper1(request, content);
            addUserInfo(requestWrapper1);
            filterChain.doFilter(requestWrapper1, response);
        }
    }

    /**
     * 校验签名
     *
     * @return 签名验证结果
     * @throws IOException 如果读取请求体失败
     */
    public String verifySignature(String body) throws IOException {
        // post请求body
        CommonRequestDTO commonRequestDTO = objectMapper.readValue(body, CommonRequestDTO.class);

        // 签名
        String appId = commonRequestDTO.getAppId();
        // 签名
        String sign = commonRequestDTO.getSign();
        // 随机数
        String nonce = commonRequestDTO.getNonce();
        // 时间戳
        String timestampStr = commonRequestDTO.getTimestamp();

        String deptCode = commonRequestDTO.getDeptCode();

        if (!StringUtils.hasText(appId) || !StringUtils.hasText(sign) || !StringUtils.hasText(nonce) || !StringUtils.hasText(timestampStr)) {
            logger.warn("缺少参数appId:{},sign:{},nonce{},timestampStr:{}", appId, sign, nonce, timestampStr);
            return "请求缺少参数,appId:" + appId + ",sign:" + sign + ",nonce:" + nonce + ",timestamp:" + timestampStr;
        }

        // 验证nonce和timestamp合法性
        if (!RedisKeyValidator.isValidString(nonce)) {
            return "不是合法的由数字和字母以及下划线组成的nonce：" + nonce;
        }
        if (!RedisKeyValidator.isValidTimestamp(timestampStr)) {
            return "不是合法的十位秒级时间戳timestamp：" + timestampStr;
        }

        // timestamp 10分钟内有效
        long timestamp = Long.parseLong(timestampStr);
        long currentTimestamp = System.currentTimeMillis() / 1000;
        if (Math.abs(currentTimestamp - timestamp) > 600) {
            return "请求已过期";
        }

//         防止请求重放，nonce只能用一次，放在redis中，有效期 20分钟
        String nonceKey = "api_signature:nonce:" + nonce;
        if (Boolean.FALSE.equals(this.redisTemplate.opsForValue().setIfAbsent(nonceKey, "1", 20, TimeUnit.MINUTES))) {
            return "nonce无效：" + nonce;
        }

        // 校验appId
        appStore = this.appStoreService.getAppSecretInfo(appId);
        String appSecret = appStore.getAppSecret();
        if (!StringUtils.hasText(appSecret)) {
            return "appId无效：" + appId;
        }

        //验证单位
        if (deptCode != null && !deptCode.startsWith(appStore.getDeptCode().replaceAll("0+$", ""))) {
            return "AppId和传入的单位不匹配,appId:" + appId + ",deptCode:" + deptCode;
        }

        String data = String.format("%s%s%s%s%s", appId, nonce, timestampStr, commonRequestDTO.getContent(), deptCode);
        String generatedSignature = DigestUtil.md5Hex(data);
        if (!generatedSignature.equals(sign)) {
            logger.warn("签名有误,generatedSignature:{},sign:{},appId:{},nonce:{},timestampStr:{},deptCode:{}", generatedSignature, sign, appId, nonce, timestampStr, deptCode);
            return "签名有误,sign: " + sign + ",appId:" + appId + ",nonce:" + nonce + ",timestamp:" + timestampStr + ",deptCode:" + deptCode;
        }
        // 签名验证通过
        return null;
    }

    /**
     * 向客户端写入响应信息
     *
     * @param response HTTP响应
     * @param msg      响应信息
     * @throws IOException 如果写入失败
     */
    private void write(HttpServletResponse response, String msg) throws IOException {
        HttpResultState httpResultState = HttpResultState.INVALID_SIGNATURE;
        httpResultState.setMessage(msg);
        HttpResult<String> httpResult = new HttpResult<>(httpResultState);
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.getWriter().write(JSONUtil.toJsonStr(httpResult));
    }

    /**
     * 增加用户信息
     *
     * @param request
     */
    private void addUserInfo(HttpServletRequest request) {
        // 增加用户信息
        String userId = request.getHeader(SignatureUtil.APPID);
        String ip = request.getRemoteAddr();
        CurrentUserInfo currentUserInfo = new CurrentUserInfo.CurrentUserInfoBuilder().userId(userId).userName(userId).ip(ip).build();
        UserContextHolder.set(currentUserInfo);
    }

    private String getRequestBody(HttpServletRequest request) throws IOException {
        try (BufferedReader reader = request.getReader()) {
            return reader.lines().collect(Collectors.joining(System.lineSeparator()));
        }
    }
}

