package com.zksy.szpt.aspect;


import com.zksy.szpt.domain.HttpResult;
import com.zksy.szpt.domain.HttpResultState;
import com.zksy.szpt.exception.NotificationException;
import org.apache.ibatis.javassist.NotFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.method.annotation.MethodArgumentTypeMismatchException;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolationException;

import java.sql.SQLIntegrityConstraintViolationException;
import java.util.stream.Collectors;


/**
 * Created by wcy on 2024/12/26.
 * 全局异常处理，包括404
 */
@ControllerAdvice
public class GlobalExceptionHandler {

    private static final Logger logger = LoggerFactory.getLogger(GlobalExceptionHandler.class);

    //一般错误
    @ExceptionHandler(value = Exception.class)
    @ResponseBody
    public HttpResult<HttpResultState> defaultErrorHandle(HttpServletRequest req, Exception e) {
        logger.error("错误信息如下：", e);
        HttpResultState state = HttpResultState.ERROR;
        return new HttpResult<>(state);
    }

    //请求映射问题
    @ExceptionHandler(value = HttpMediaTypeNotSupportedException.class)
    @ResponseBody
    public HttpResult<HttpResultState> handleHttpMediaTypeNotSupportedException(HttpServletRequest req, Exception e) {
        logger.error("错误信息如下：", e);
        HttpResultState state = HttpResultState.NOTIFICATION;
        state.setMessage(e.getMessage());
        return new HttpResult<>(state);
    }

    //not found
    @ResponseBody
    @ExceptionHandler(value = {NotFoundException.class, NoHandlerFoundException.class})
    public HttpResult<HttpResultState> handlerNotFoundException(HttpServletRequest req, Exception e) {
        HttpResultState state = HttpResultState.NOT_FOUND;
        state.setMessage("请求的资源" + req.getRequestURL() + "不存在");
        return new HttpResult<>(state);
    }

    //region 输入参数相关的异常
    //valid注解
    //（Bean 校验异常）
    @ExceptionHandler(value = MethodArgumentNotValidException.class)
    @ResponseBody
    public HttpResult<HttpResultState> handlerMethodArgumentNotValidException(HttpServletRequest req, MethodArgumentNotValidException e) {
        logger.warn(e.getMessage());
        HttpResultState state = HttpResultState.INVALID_PARAMETER;
        state.setMessage(e.getBindingResult().getFieldErrors().stream().map(k -> k.getField() + "[" + k.getRejectedValue() + "]" + k.getDefaultMessage()).collect(Collectors.joining(";")));
        return new HttpResult<>(state);
    }

    //validate注解
    //（方法参数校验异常）如实体类中的@Size注解配置和数据库中该字段的长度不统一等问题
    @ExceptionHandler(value = ConstraintViolationException.class)
    @ResponseBody
    public HttpResult<HttpResultState> handlerConstraintViolationException(HttpServletRequest req, ConstraintViolationException e) {
        logger.warn(e.getMessage());
        HttpResultState state = HttpResultState.INVALID_PARAMETER;
        state.setMessage(e.getMessage());
        return new HttpResult<>(state);
    }

    //RequestBody反序列化失败
    @ResponseBody
    @ExceptionHandler({HttpMessageNotReadableException.class})
    public HttpResult<HttpResultState> messageExceptionHandler(HttpMessageNotReadableException e) {
        logger.warn("http请求参数转换异常: " + e.getMessage());
        HttpResultState state = HttpResultState.INVALID_PARAMETER;
        state.setMessage("请求参数不正确");
        return new HttpResult<>(state);
    }

    //RequestParam参数缺失
    @ResponseBody
    @ExceptionHandler({MissingServletRequestParameterException.class})
    public HttpResult<HttpResultState> MissParameterExceptionHandler(MissingServletRequestParameterException e) {
        logger.warn(e.getMessage());
        HttpResultState state = HttpResultState.INVALID_PARAMETER;
        state.setMessage(e.getMessage());
        return new HttpResult<>(state);
    }

    //RequestParam参数类型错误
    @ResponseBody
    @ExceptionHandler({MethodArgumentTypeMismatchException.class})
    public HttpResult<HttpResultState> MethodArgumentTypeMismatchExceptionHandler(MethodArgumentTypeMismatchException e) {
        logger.warn(e.getMessage());
        HttpResultState state = HttpResultState.INVALID_PARAMETER;
        state.setMessage(e.getCause().getMessage());
        return new HttpResult<>(state);
    }

//    @ResponseBody
//    @ExceptionHandler({ParameterDataTypeException.class})
//    public HttpResult<HttpResultState> ParameterDataTypeExceptionHandler(ParameterDataTypeException e) {
//        logger.warn(e.getMessage());
//        HttpResultState state = HttpResultState.INVALID_PARAMETER;
//        state.setMessage(e.getMessage());
//        return new HttpResult<>(state);
//    }
    //endregion

    //region 数据库相关异常
    //处理数据层面的顶级异常
    @ResponseBody
    @ExceptionHandler({DataAccessException.class})
    public HttpResult<HttpResultState> DataAccessExceptionHandler(DataAccessException e) {
        logger.error(e.getMessage());
        HttpResultState state = HttpResultState.DATABASE;
        return new HttpResult<>(state);
    }

    //违反唯一约束
    @ResponseBody
    @ExceptionHandler({DuplicateKeyException.class})
    public HttpResult<HttpResultState> DuplicateKeyExceptionHandler(DuplicateKeyException e) {
        logger.warn(e.getMessage());
        HttpResultState state = HttpResultState.DUPLICATED_KEY;
        state.setMessage(e.getCause().getMessage());
        return new HttpResult<>(state);
    }

    //违反唯一约束
    @ResponseBody
    @ExceptionHandler({SQLIntegrityConstraintViolationException.class})
    public HttpResult<HttpResultState> DuplicateKeyExceptionHandler(SQLIntegrityConstraintViolationException e) {
        logger.warn(e.getMessage());
        HttpResultState state = HttpResultState.DUPLICATED_KEY;
        state.setMessage(e.getCause().getMessage());
        return new HttpResult<>(state);
    }

    //endregion


    @ResponseBody
    @ExceptionHandler({NotificationException.class})
    public HttpResult<HttpResultState> NotificationExceptionHandler(NotificationException e) {
        HttpResultState state = HttpResultState.NOTIFICATION;
        state.setMessage(e.getMessage());
        return new HttpResult<>(state);
    }


}
