package com.zksy.szpt.filter;


import cn.hutool.crypto.digest.DigestUtil;
import cn.hutool.json.JSONUtil;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.zksy.szpt.domain.CurrentUserInfo;
import com.zksy.szpt.domain.HttpResult;
import com.zksy.szpt.domain.HttpResultState;
import com.zksy.szpt.domain.po.AppStore;
import com.zksy.szpt.service.AppStoreService;
import com.zksy.szpt.util.EncryptUtil;
import com.zksy.szpt.util.RedisKeyValidator;
import com.zksy.szpt.util.SignatureUtil;
import com.zksy.szpt.util.UserContextHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.annotation.Resource;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * body {单位， 房间， 人员信息数组}  请求头加nonce timestamp appId sign(body[遍历按照key按照顺序&拼接] 加 nonece timestamp appId 用$符号拼接)
 */
@Component
public class SignatureVerificationFilter extends OncePerRequestFilter {

    public Logger logger = LoggerFactory.getLogger(SignatureVerificationFilter.class);

    @Resource
    ObjectMapper objectMapper;

    @Resource
    private RedisTemplate<String, Object> redisTemplate;

    private final AppStoreService appStoreService;

    public SignatureVerificationFilter(AppStoreService appStoreService) {
        this.appStoreService = appStoreService;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        // 对request进行包装，支持重复读取body
        ReusableBodyRequestWrapper requestWrapper = new ReusableBodyRequestWrapper(request);
        // 校验签名
        if (this.verifySignature(requestWrapper, response)) {
            filterChain.doFilter(requestWrapper, response);
        }
    }

    /**
     * 校验签名
     *
     * @param request  HTTP请求
     * @param response HTTP响应
     * @return 签名验证结果
     * @throws IOException 如果读取请求体失败
     */
    public boolean verifySignature(HttpServletRequest request, HttpServletResponse response) throws IOException {
        // 签名
        String appId = request.getHeader(SignatureUtil.APPID);
        // 签名
        String sign = request.getHeader(SignatureUtil.SIGNATURE);
        // 随机数
        String nonce = request.getHeader(SignatureUtil.NONCE);
        // 时间戳
        String timestampStr = request.getHeader(SignatureUtil.TIMESTAMP);
        // 单位
        String deptCode = request.getHeader(SignatureUtil.DEPT_CODE);

        if (!StringUtils.hasText(appId) || !StringUtils.hasText(sign) || !StringUtils.hasText(nonce) || !StringUtils.hasText(timestampStr) || !StringUtils.hasText(deptCode)) {
            logger.warn("缺少参数appId:{},sign:{},nonce{},timestampStr:{},deptCode:{}", appId, sign, nonce, timestampStr, deptCode);
            this.write(response, "请求头缺少参数,appId:" + appId + ",sign:" + sign + ",nonce:" + nonce + ",timestamp:" + timestampStr + ",deptCode:" + deptCode);
            return false;
        }

        // 验证nonce和timestamp合法性
        if (!RedisKeyValidator.isValidString(nonce)) {
            this.write(response, "不是合法的由数字和字母以及下划线组成的nonce：" + nonce);
            return false;
        }
        if (!RedisKeyValidator.isValidTimestamp(timestampStr)) {
            this.write(response, "不是合法的十位秒级时间戳timestamp：" + timestampStr);
            return false;
        }

        // timestamp 10分钟内有效
        long timestamp = Long.parseLong(timestampStr);
        long currentTimestamp = System.currentTimeMillis() / 1000;
        if (Math.abs(currentTimestamp - timestamp) > 600) {
            this.write(response, "请求已过期");
            return false;
        }

//         防止请求重放，nonce只能用一次，放在redis中，有效期 20分钟
        String nonceKey = "api_signature:nonce:" + nonce;
        if (Boolean.FALSE.equals(this.redisTemplate.opsForValue().setIfAbsent(nonceKey, "1", 20, TimeUnit.MINUTES))) {
            this.write(response, "nonce无效：" + nonce);
            return false;
        }

        // 校验appId
        AppStore appStore = this.appStoreService.getAppSecretInfo(appId);
        String appSecret = appStore.getAppSecret();
        if (!StringUtils.hasText(appSecret)) {
            this.write(response, "appId无效：" + appId);
            return false;
        }

        // post请求body
        String body = StreamUtils.copyToString(request.getInputStream(), StandardCharsets.UTF_8);
        Map<String, Object> objectMap = objectMapper.readValue(body, Map.class);

        //验证单位
        if (!deptCode.startsWith(appStore.getDeptCode().replaceAll("0+$", ""))) {
            write(response, "AppId和传入的单位不匹配,appId:" + appId + ",deptCode:" + objectMap.get("deptCode"));
            return false;
        }

        body = objectMapper.writeValueAsString(objectMap);
        logger.info("请求参数appId: {}, nonce: {}, timestampStr: {}, 原始body: {}, deptCode: {}", appId, nonce, timestampStr, body, deptCode);
        body = EncryptUtil.getInstance().AESEncode(body, appSecret);
//        logger.info("appSecret{}加密后body: {}", appSecret,body);
        // 校验签名appId+nonce+timestampStr+aes(body,secret)+detCode
        String data = String.format("%s%s%s%s%s", appId, nonce, timestampStr, body, deptCode);
        String generatedSignature = DigestUtil.md5Hex(data);
        if (!generatedSignature.equals(sign)) {
            logger.warn("签名有误,generatedSignature:{},sign:{},appId:{},nonce:{},timestampStr:{},deptCode:{}", generatedSignature, sign, appId, nonce, timestampStr, deptCode);
            write(response, "签名有误,sign: " + sign + ",appId: " + appId + ",nonce: " + nonce + ",timestamp: " + timestampStr + ",deptCode:" + deptCode);
            return false;
        }

        // 签名验证通过
        addUserInfo(request);
        return true;
    }

    /**
     * 向客户端写入响应信息
     *
     * @param response HTTP响应
     * @param msg      响应信息
     * @throws IOException 如果写入失败
     */
    private void write(HttpServletResponse response, String msg) throws IOException {
        HttpResultState httpResultState = HttpResultState.INVALID_SIGNATURE;
        httpResultState.setMessage(msg);
        HttpResult<String> httpResult = new HttpResult<>(httpResultState);
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.getWriter().write(JSONUtil.toJsonStr(httpResult));
    }

    /**
     * 增加用户信息
     *
     * @param request
     */
    private void addUserInfo(HttpServletRequest request) {
        // 增加用户信息
        String userId = request.getHeader(SignatureUtil.APPID);
        String ip = request.getRemoteAddr();
        CurrentUserInfo currentUserInfo = new CurrentUserInfo.CurrentUserInfoBuilder().userId(userId).userName(userId).ip(ip).deptCode(request.getHeader(SignatureUtil.DEPT_CODE)).build();
        UserContextHolder.set(currentUserInfo);
    }
}

