package com.zksy.szpt.filter;


import cn.hutool.crypto.digest.DigestUtil;
import cn.hutool.json.JSONUtil;
import com.zksy.szpt.domain.HttpResult;
import com.zksy.szpt.domain.HttpResultState;
import com.zksy.szpt.service.AppStoreService;
import com.zksy.szpt.util.SignatureUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;


@Component
public class SignatureVerificationFilter extends OncePerRequestFilter {

    public Logger logger = LoggerFactory.getLogger(SignatureVerificationFilter.class);

    private static Map<String, Function<String, String>> workTypeMap = new HashMap<>();

    private final AppStoreService appStoreService;

    public SignatureVerificationFilter(AppStoreService appStoreService) {
        this.appStoreService = appStoreService;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        // 对request进行包装，支持重复读取body
        ReusableBodyRequestWrapper requestWrapper = new ReusableBodyRequestWrapper(request);
        // 校验签名
        if (this.verifySignature(requestWrapper, response)) {
            filterChain.doFilter(requestWrapper, response);
        }
    }

    /**
     * 校验签名
     *
     * @param request  HTTP请求
     * @param response HTTP响应
     * @return 签名验证结果
     * @throws IOException 如果读取请求体失败
     */
    public boolean verifySignature(HttpServletRequest request, HttpServletResponse response) throws IOException {
        // 签名
        String appId = request.getHeader(SignatureUtil.APPID);
        // 签名
        String sign = request.getHeader(SignatureUtil.SIGNATURE);
        // 随机数
        String nonce = request.getHeader(SignatureUtil.NONCE);
        // 时间戳
        String timestampStr = request.getHeader(SignatureUtil.TIMESTAMP);


        if (!StringUtils.hasText(appId) || !StringUtils.hasText(sign) || !StringUtils.hasText(nonce) || !StringUtils.hasText(timestampStr)) {
            logger.error("缺少参数appId{},sign{},nonce{},timestampStr{}", appId, sign, nonce, timestampStr);
            this.write(response, "缺少参数");
            return false;
        }

        String secretKey = this.appStoreService.getAppSecretByAppKey(appId);

        // timestamp 10分钟内有效
//        long timestamp = Long.parseLong(timestampStr);
//        long currentTimestamp = System.currentTimeMillis() / 1000;
//        if (Math.abs(currentTimestamp - timestamp) > 600) {
//            this.write(response, "请求已过期");
//            return false;
//        }

        // 防止请求重放，nonce只能用一次，放在redis中，有效期 20分钟
//        String nonceKey = "SignatureVerificationFilter:nonce:" + nonce;
//        if (!this.redisTemplate.opsForValue().setIfAbsent(nonceKey, "1", 20, TimeUnit.MINUTES)) {
//            this.write(response, "nonce无效");
//            return false;
//        }

        // 请求体
        String body = StreamUtils.copyToString(request.getInputStream(), StandardCharsets.UTF_8);
        // 需要签名的数据：appId+nonce+timestampStr+body+secretKey
        // 校验签名
        String data = String.format("%s%s%s%s%s", appId, nonce, timestampStr, body, secretKey);
        String generatedSignature = DigestUtil.md5Hex(data);
        if (!generatedSignature.equals(sign)) {
            write(response, "签名有误");
            return false;
        }
        return true;
    }

    /**
     * 向客户端写入响应信息
     *
     * @param response HTTP响应
     * @param msg      响应信息
     * @throws IOException 如果写入失败
     */
    private void write(HttpServletResponse response, String msg) throws IOException {
        HttpResultState httpResultState = HttpResultState.INVALID_TOKEN;
        httpResultState.setMessage(msg);
        HttpResult<String> httpResult = new HttpResult<>(httpResultState);
        response.setContentType(MediaType.APPLICATION_JSON_VALUE);
        response.setCharacterEncoding(StandardCharsets.UTF_8.name());
        response.getWriter().write(JSONUtil.toJsonStr(httpResult));
    }
}

