package com.cx.cn.cxquartz.service.quartz.impl;


import com.cx.cn.cxquartz.bean.*;
import com.cx.cn.cxquartz.job.QuartzMainJobFactory;
import com.cx.cn.cxquartz.service.quartz.QuartzService;
import com.cx.cn.cxquartz.service.quartz.QuartzTaskErrorsService;
import com.cx.cn.cxquartz.service.quartz.QuartzTaskInformationsService;
import com.cx.cn.cxquartz.service.quartz.QuartzTaskRecordsService;
import com.cx.cn.cxquartz.util.ResultEnum;
import com.cx.cn.cxquartz.util.ResultUtil;
import com.cx.cn.cxquartz.vo.QuartzTaskRecordsVo;
import org.apache.commons.collections.CollectionUtils;
import org.quartz.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.scheduling.quartz.SchedulerFactoryBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

@Service
@Configuration
public class QuartzServiceImpl implements QuartzService, InitializingBean {

    private static final Logger logger = LoggerFactory.getLogger(QuartzServiceImpl.class);
    public static final String QUARTZ_TOPIC = "quartztopic";

    private AtomicInteger atomicInteger;

    @Autowired
    private KafkaTemplate kafkaTemplate;

    @Autowired
    private QuartzTaskInformationsService quartzTaskInformationsService;

    @Autowired
    private QuartzTaskErrorsService quartzTaskErrorsService;

    @Autowired
    private QuartzTaskRecordsService quartzTaskRecordsService;

    @Autowired
    private SchedulerFactoryBean schedulerBean;


    /**
     * 列表查询所有定时任务
     *
     * @return
     */
    @Override
    public List<QuartzTaskInformations> getTaskList(String taskNo, String currentPage) {
        List<QuartzTaskInformations> quartzTaskInformations = quartzTaskInformationsService.selectList(taskNo, currentPage);
        return quartzTaskInformations;
    }

    /**
     * 新增定时任务
     *
     * @param quartzTaskInformations
     * @return
     */
    @Override
    public String addTask(QuartzTaskInformations quartzTaskInformations) {
        String result = quartzTaskInformationsService.insert(quartzTaskInformations);
        return result;
    }

    @Override
    public QuartzTaskInformations getTaskById(String id) {
        return quartzTaskInformationsService.getTaskById(id);
    }

    @Override
    public String updateTask(QuartzTaskInformations quartzTaskInformations) {
        return quartzTaskInformationsService.updateTask(quartzTaskInformations);
    }

    @Override
    public Integer UpdateExecuteParam(Map map) {
        return  quartzTaskInformationsService.UpdateExecuteParam(map);
    }

    /**
     * 启动 or 暂停定时任务
     *
     * @param taskNo
     * @return
     * @throws SchedulerException
     */
    @Override
    @Transactional
    public String startJob(String taskNo,String status) throws SchedulerException {
        QuartzTaskInformations quartzTaskInformation = quartzTaskInformationsService.getTaskByTaskNo(taskNo);
        if (quartzTaskInformation == null) {
            return ResultUtil.success(ResultEnum.NO_DATA.getCode(), ResultEnum.NO_DATA.getMessage());
        }
        //String status = quartzTaskInformation.getFrozenstatus();
        Scheduler scheduler = schedulerBean.getScheduler();
        long currentTimeMillis = System.currentTimeMillis();
        QuartzTaskInformations task = new QuartzTaskInformations();
        task.setId(quartzTaskInformation.getId());
        task.setVersion(quartzTaskInformation.getVersion());
        //说明要暂停
        if (ResultEnum.FROZEN.name().equals(status)) {
            //删除
            //暂停
            scheduler.pauseTrigger(new TriggerKey(taskNo));
            // 移除触发器中的任务
            scheduler.unscheduleJob(new TriggerKey(taskNo));
            scheduler.deleteJob(new JobKey(taskNo));
            task.setFrozentime(currentTimeMillis);
            task.setFrozenstatus(ResultEnum.FROZEN.name());
        } else if (ResultEnum.UNFROZEN.name().equals(status)) {
            //删除后启动
            //暂停
            scheduler.pauseTrigger(new TriggerKey(taskNo));
            // 移除触发器中的任务
            scheduler.unscheduleJob(new TriggerKey(taskNo));
            scheduler.deleteJob(new JobKey(taskNo));
            this.schedule(quartzTaskInformation, scheduler);
            task.setUnfrozentime(currentTimeMillis);
            task.setFrozenstatus(ResultEnum.UNFROZEN.name());
        }
       // task.setLastmodifytime(currentTimeMillis);
        //quartzTaskInformationsService.updateStatusById(task);
       // logger.info("taskNo={},taskName={},scheduleRule={},任务{}成功", quartzTaskInformation.getTaskno(), quartzTaskInformation.getTaskname(), quartzTaskInformation.getSchedulerrule(), ResultEnum.FROZEN.name().equals(status) ? "启动" : "暂停");
        return ResultUtil.success(ResultEnum.SUCCESS.getCode(), ResultEnum.SUCCESS.getMessage());
    }

    public String startOrStopJob(String taskNo,String status) throws SchedulerException {
        QuartzTaskInformations quartzTaskInformation = quartzTaskInformationsService.getTaskByTaskNo(taskNo);
        if (quartzTaskInformation == null) {
            return ResultUtil.success(ResultEnum.NO_DATA.getCode(), ResultEnum.NO_DATA.getMessage());
        }
        Scheduler scheduler = schedulerBean.getScheduler();
        long currentTimeMillis = System.currentTimeMillis();
        QuartzTaskInformations task = new QuartzTaskInformations();
        task.setId(quartzTaskInformation.getId());
        task.setVersion(quartzTaskInformation.getVersion());
        //暂停
        scheduler.pauseTrigger(new TriggerKey(taskNo));
        // 移除触发器中的任务
        scheduler.unscheduleJob(new TriggerKey(taskNo));
        scheduler.deleteJob(new JobKey(taskNo));
        task.setFrozentime(currentTimeMillis);
        if (ResultEnum.FROZEN.name().equals(status)) {
            task.setFrozenstatus(ResultEnum.FROZEN.name());
            //启动
        } else if (ResultEnum.UNFROZEN.name().equals(status)) {
            this.schedule(quartzTaskInformation, scheduler);
            task.setFrozenstatus(ResultEnum.UNFROZEN.name());
        }
        task.setLastmodifytime(currentTimeMillis);
        quartzTaskInformationsService.updateStatusById(task);
        logger.info("taskNo={},taskName={},scheduleRule={},任务{}成功", quartzTaskInformation.getTaskno(), quartzTaskInformation.getTaskname(), quartzTaskInformation.getSchedulerrule(), ResultEnum.FROZEN.name().equals(status) ? "启动" : "暂停");
        return ResultUtil.success(ResultEnum.SUCCESS.getCode(), ResultEnum.SUCCESS.getMessage());
    }
    /**
     * 初始化加载定时任务
     *
     * @throws Exception
     */
    @Override
    public void initLoadOnlineTasks() {
        //判断是否是分发任务的配置


        List<QuartzTaskInformations> unnfrozenTasks = quartzTaskInformationsService.getUnnfrozenTasks(ResultEnum.UNFROZEN.name());
        if (CollectionUtils.isEmpty(unnfrozenTasks)) {
            logger.info("没有需要初始化加载的定时任务");
            return;
        }
        Scheduler scheduler = schedulerBean.getScheduler();
        for (QuartzTaskInformations unnfrozenTask : unnfrozenTasks) {
            try {
                this.schedule(unnfrozenTask, scheduler);
            } catch (Exception e) {
                logger.error("系统初始化加载定时任务:taskno={},taskname={}失败原因exception={}", unnfrozenTask.getTaskno(), unnfrozenTask.getTaskname(), e);
            }
        }
    }

    /**
     * 初始化加载定时任务
     *
     * @throws Exception
     */
    @Override
    public void afterPropertiesSet() throws Exception {
            this.initLoadOnlineTasks();
    }

    @Override
    public QuartzTaskRecords addTaskRecords(String taskNo) {
        QuartzTaskRecords quartzTaskRecords = null;
        try {
            QuartzTaskInformations quartzTaskInformation = quartzTaskInformationsService.getTaskByTaskNo(taskNo);
            if (null == quartzTaskInformation || ResultEnum.FROZEN.name().equals(quartzTaskInformation.getFrozenstatus())) {
                logger.info("taskNo={} not exist or status is frozen!");
                return null;
            }
            long currentTimeMillis = System.currentTimeMillis();
            //QuartzTaskInformations task = new QuartzTaskInformations();
            //task.setId(quartzTaskInformation.getId());
           // task.setLastmodifytime(currentTimeMillis);
           // quartzTaskInformationsService.updateTask(task);
            //logger.info("taskNo={},taskName={}更新最后修改时间成功", quartzTaskInformation.getTaskno(), quartzTaskInformation.getTaskname());

            quartzTaskRecords = new QuartzTaskRecords();
            quartzTaskRecords.setTaskno(taskNo);
            quartzTaskRecords.setTimekeyvalue(quartzTaskInformation.getTimekey());
            quartzTaskRecords.setExecutetime(currentTimeMillis);
            quartzTaskRecords.setTaskstatus(ResultEnum.INIT.name());
            quartzTaskRecords.setFailcount(0);
            quartzTaskRecords.setFailreason("");
            quartzTaskRecords.setCreatetime(currentTimeMillis);
            quartzTaskRecords.setLastmodifytime(currentTimeMillis);
            quartzTaskRecordsService.addTaskRecords(quartzTaskRecords);
            logger.info("taskNo={},taskName={}添加执行记录表成功", quartzTaskInformation.getTaskno(), quartzTaskInformation.getTaskname());
        } catch (Exception ex) {
            logger.error("添加执行记录表异常exceptio={}", ex);
            return null;
        }
        return quartzTaskRecords;
    }

    @Override
    public Integer updateRecordById(Integer count, Long id) {
        QuartzTaskRecords records = new QuartzTaskRecords();
        records.setId(id);
        records.setFailcount(count);
        records.setLastmodifytime(System.currentTimeMillis());
        if (count > 0) {
            records.setTaskstatus(ResultEnum.FAIL.name());
        } else {
            records.setTaskstatus(ResultEnum.SUCCESS.name());
        }
        return quartzTaskRecordsService.updateTaskRecords(records);
    }

    @Override
    public Integer updateModifyTimeById(QuartzTaskInformations quartzTaskInformations) {
        return quartzTaskInformationsService.updateModifyTimeById(quartzTaskInformations);
    }

    @Override
    public Integer addTaskErrorRecord(String id, String errorKey, String errorValue) {
        QuartzTaskErrors taskErrors = new QuartzTaskErrors();
        taskErrors.setTaskexecuterecordid(String.valueOf(id));
        taskErrors.setErrorkey(errorKey);
        taskErrors.setCreatetime(System.currentTimeMillis());
        taskErrors.setLastmodifytime(System.currentTimeMillis());
        taskErrors.setErrorvalue(errorValue);
        return quartzTaskErrorsService.addTaskErrorRecord(taskErrors);
    }

    public void schedule(QuartzTaskInformations quartzTaskInfo, Scheduler scheduler) throws SchedulerException {
        TriggerKey triggerKey = TriggerKey.triggerKey(quartzTaskInfo.getTaskno(), Scheduler.DEFAULT_GROUP);
        JobDetail jobDetail = JobBuilder.newJob(QuartzMainJobFactory.class).withDescription(quartzTaskInfo.getTaskname()).withIdentity(quartzTaskInfo.getTaskno(), Scheduler.DEFAULT_GROUP).build();
        JobDataMap jobDataMap = jobDetail.getJobDataMap();
        jobDataMap.put("taskinfo", quartzTaskInfo);
//        jobDataMap.put("taskNo", quartzTaskInfo.getTaskno());
//        jobDataMap.put("executorNo", quartzTaskInfo.getExecutorno());
////        jobDataMap.put("sendType", quartzTaskInfo.getSendtype());
//        jobDataMap.put("url", quartzTaskInfo.getUrl());
//        jobDataMap.put("executeParameter", quartzTaskInfo.getExecuteparamter());
        CronScheduleBuilder cronScheduleBuilder = CronScheduleBuilder.cronSchedule(quartzTaskInfo.getSchedulerrule());
        CronTrigger cronTrigger = TriggerBuilder.newTrigger().withDescription(quartzTaskInfo.getTaskname()).withIdentity(triggerKey).withSchedule(cronScheduleBuilder).build();
        scheduler.scheduleJob(jobDetail, cronTrigger);
//        logger.info("taskNo={},taskName={},scheduleRule={} load to quartz success!", quartzTaskInfo.getTaskno(), quartzTaskInfo.getTaskname(), quartzTaskInfo.getSchedulerrule());
    }

    /**
     * kafka推送消息
     *
     * @param message
     */
    @Override
    public void sendMessage(String message) {
        kafkaTemplate.send(QUARTZ_TOPIC, message);
        logger.info("给kafka推送消息message={}成功", message);
    }

    @Override
    public List<QuartzTaskRecordsVo> taskRecords(String taskNo) {
        List<QuartzTaskRecords> quartzTaskRecords = quartzTaskRecordsService.listTaskRecordsByTaskNo(taskNo);
        QuartzTaskRecordsVo recordsVo = null;
        List<QuartzTaskRecordsVo> voList = new ArrayList<>();
        for (QuartzTaskRecords quartzTaskRecord : quartzTaskRecords) {
            recordsVo = new QuartzTaskRecordsVo();
            BeanUtils.copyProperties(quartzTaskRecord, recordsVo);
            recordsVo.setTime(quartzTaskRecord.getLastmodifytime() - quartzTaskRecord.getCreatetime());
            voList.add(recordsVo);
        }
        return voList;
    }

    @Override
    public String runTaskRightNow(String taskNo) {
        return null;
    }


    @Override
    public QuartzTaskErrors detailTaskErrors(String recordId) {
        return quartzTaskErrorsService.detailTaskErrors(recordId);
    }

    @Override
    public List<QuartzTaskInformations> getTaskSByDeviceCode(String devicecode) {
        return quartzTaskInformationsService.getTaskSByDeviceCode(devicecode);
    }

    @Override
    public Integer deletetask(String taskno) {
        return quartzTaskInformationsService.deletetask(taskno);
    }
}

