package im.dx.system.controller;

import com.github.pagehelper.PageInfo;
import im.dx.common.util.PageResultBean;
import im.dx.common.util.ResultBean;
import im.dx.system.model.Role;
import im.dx.system.model.TaskParams;
import im.dx.system.service.RoleService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@Controller
@RequestMapping("/role")
public class RoleController {

    @Autowired
    private RoleService roleService;

    @GetMapping("/index")
    public String index() {
        return "role/role-list";
    }

    /**
     * 查询角色列表
     * @param page
     * @param limit
     * @param roleQuery
     * @return
     */
    @GetMapping("/list")
    @ResponseBody
    public PageResultBean<Role> getList(@RequestParam(value = "page", defaultValue = "1") int page,
                                        @RequestParam(value = "limit", defaultValue = "10")int limit,
                                        Role roleQuery) {
        List<Role> roles = roleService.selectAll(page, limit, roleQuery);
        PageInfo<Role> rolePageInfo = new PageInfo<>(roles);
        return new PageResultBean<>(rolePageInfo.getTotal(), rolePageInfo.getList());
    }

    @GetMapping
    public String add() {
        return "role/role-add";
    }

    /**
     * 新增角色
     * @param role
     * @return
     */
    @PostMapping
    @ResponseBody
    public ResultBean add(Role role) {
        roleService.add(role);
        return ResultBean.success();
    }

    @GetMapping("/{roleId}")
    public String update(@PathVariable("roleId") Integer roleId, Model model) {
        Role role = roleService.selectOne(roleId);
        model.addAttribute("role", role);
        return "role/role-add";
    }

    /**
     * 修改角色
     * @param role
     * @return
     */
    @PutMapping
    @ResponseBody
    public ResultBean update(Role role) {
        roleService.update(role);
        return ResultBean.success();
    }

    /***
     * 删除角色
     * @param roleId
     * @return
     */
    @DeleteMapping("/{roleId}")
    @ResponseBody
    public ResultBean delete(@PathVariable("roleId") Integer roleId) {
        roleService.delete(roleId);
        return ResultBean.success();
    }

    /***
     * 为角色授予菜单
     * @param roleId
     * @param menuIds
     * @return
     */
    @PostMapping("/{roleId}/grant/menu")
    @ResponseBody
    public ResultBean grantMenu(@PathVariable("roleId") Integer roleId, @RequestParam(value = "menuIds[]", required = false) Integer[] menuIds) {
        roleService.grantMenu(roleId, menuIds);
        return ResultBean.success();
    }

    /***
     * 为角色授予操作权限
     * @param roleId
     * @param operatorIds
     * @return
     */
    @PostMapping("/{roleId}/grant/operator")
    @ResponseBody
    public ResultBean grantOperator(@PathVariable("roleId") Integer roleId, @RequestParam(value = "operatorIds[]", required = false) Integer[] operatorIds) {
        roleService.grantOperator(roleId, operatorIds);
        return ResultBean.success();
    }

    /**
     * 获取角色拥有的菜单
     */
    @GetMapping("/{roleId}/own/menu")
    @ResponseBody
    public ResultBean getRoleOwnMenu(@PathVariable("roleId") Integer roleId) {
        return ResultBean.success(roleService.getMenusByRoleId(roleId));
    }

    /**
     * 获取角色拥有的操作权限
     */
    @GetMapping("/{roleId}/own/operator")
    @ResponseBody
    public ResultBean getRoleOwnOperator(@PathVariable("roleId") Integer roleId) {
        Integer[] operatorIds = roleService.getOperatorsByRoleId(roleId);
        for (int i = 0; i < operatorIds.length; i++) {
            operatorIds[i] = operatorIds[i] + 10000;
        }
        return ResultBean.success(operatorIds);
    }

    /***
     * 为角色授予菜单
     * @param roleId
     * @param videorecordIds
     * @return
     */
    @PostMapping("/{roleId}/grant/videorecord")
    @ResponseBody
    public ResultBean grantVideorecord(@PathVariable("roleId") Integer roleId, @RequestParam(value = "videorecordIds[]", required = false) String[] videorecordIds) {
        roleService.grantVideorecord(roleId, videorecordIds);
        return ResultBean.success();
    }

    /***
     * 查询角色用户树形数据
     * @return
     */
    @GetMapping("/listAllUsers")
    @ResponseBody
    public ResultBean listAllUsers() {
        return ResultBean.success(roleService.queryRoleUserTree());

    }

    /***
     * 给用户添加任务
     * @param taskParams
     * @return
     */
    @PostMapping("/addtaskinfo")
    @ResponseBody
    public ResultBean addtaskinfo(@RequestBody TaskParams taskParams) {
        return ResultBean.success(roleService.addtaskinfo(taskParams));

    }
}
