package im.dx.system.model;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Getter;
import lombok.Setter;

import java.util.List;

/**
 * 自定义响应结构
 * @author  cp
 */
@Getter
@Setter
public class ResultObj {

    // 定义jackson对象
    private static final ObjectMapper MAPPER = new ObjectMapper();

    /**
     * 响应业务状态
     * 200	成功
     * 201	错误
     * 400	参数错误
     */
    private Integer status;

    /**
     * 响应消息
     */
    private String msg;

    /**
     * 响应中的数据
     */
    private Object data;

    public static ResultObj error(Integer status, String msg, Object data) {
        return new ResultObj(status, msg, data);
    }

    public static ResultObj ok(Object data) {
        return new ResultObj(data);
    }

    public static ResultObj ok() {
        return ok(null);
    }

    private ResultObj() {

    }

    public static ResultObj error(Integer status, String msg) {
        return new ResultObj(status, msg, null);
    }

    private ResultObj(Integer status, String msg, Object data) {
        this.status = status;
        this.msg = msg;
        this.data = data;
    }

    private ResultObj(Object data) {
        this.status = 200;
        this.msg = "OK";
        this.data = data;
    }


    /**
     * 将json结果集转化为SysResult对象
     *
     * @param jsonData json数据
     * @param clazz SysResult中的object类型
     * @return SysResult对象
     */
    public static ResultObj formatToPojo(String jsonData, Class<?> clazz) {
        try {
            if (clazz == null) {
                return MAPPER.readValue(jsonData, ResultObj.class);
            }
            JsonNode jsonNode = MAPPER.readTree(jsonData);
            JsonNode data = jsonNode.get("data");
            Object obj = null;
            if (data.isObject()) {
                obj = MAPPER.readValue(data.traverse(), clazz);
            } else if (data.isTextual()) {
                obj = MAPPER.readValue(data.asText(), clazz);
            }
            return error(jsonNode.get("status").intValue(), jsonNode.get("msg").asText(), obj);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 没有object对象的转化
     *
     * @param json 字符串
     * @return SysResult对象
     */
    public static ResultObj format(String json) {
        try {
            return MAPPER.readValue(json, ResultObj.class);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Object是集合转化
     *
     * @param jsonData json数据
     * @param clazz 集合中的类型
     * @return SysResult对象
     */
    public static ResultObj formatToList(String jsonData, Class<?> clazz) {
        try {
            JsonNode jsonNode = MAPPER.readTree(jsonData);
            JsonNode data = jsonNode.get("data");
            Object obj = null;
            if (data.isArray() && data.size() > 0) {
                obj = MAPPER.readValue(data.traverse(),
                        MAPPER.getTypeFactory().constructCollectionType(List.class, clazz));
            }
            return error(jsonNode.get("status").intValue(), jsonNode.get("msg").asText(), obj);
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    @Override
    public String toString() {
        return "ResultObj{" +
                "status=" + status +
                ", msg='" + msg + '\'' +
                ", data=" + data +
                '}';
    }

}